<?php

namespace WowLive\Controllers;

use WowLive\Models\Participant;
use WowLive\Models\Event;
use PhpOffice\PhpSpreadsheet\IOFactory;

class ParticipantController {
    private $db;
    private $participant;
    private $event;

    public function __construct($db) {
        $this->db = $db;
        $this->participant = new Participant($db);
        $this->event = new Event($db);
    }

    public function index() {
        $evento_id = $_GET['evento_id'] ?? null;
        $search = $_GET['search'] ?? '';
        
        // Buscar participantes
        $stmt = $this->participant->read($evento_id);
        $participantes = $stmt->fetchAll();
        
        // Filtrar por busca se necessário
        if (!empty($search)) {
            $participantes = array_filter($participantes, function($p) use ($search) {
                return stripos($p['nome_completo'], $search) !== false ||
                       stripos($p['email'], $search) !== false ||
                       stripos($p['codigo_inscricao'], $search) !== false;
            });
        }

        // Buscar eventos para o filtro
        $stmt_eventos = $this->event->read();
        $eventos = $stmt_eventos->fetchAll();

        // Buscar evento selecionado
        $evento_selecionado = null;
        if ($evento_id) {
            $this->event->findById($evento_id);
            $evento_selecionado = $this->event;
        }

        $success = $_SESSION['success'] ?? '';
        $error = $_SESSION['error'] ?? '';
        unset($_SESSION['success'], $_SESSION['error']);

        include '../src/Views/participants/index.php';
    }

    public function showForm() {
        $evento_id = $_GET['evento_id'] ?? null;
        
        // Buscar eventos
        $stmt = $this->event->read();
        $eventos = $stmt_eventos->fetchAll();

        // Gerar próximo código
        $next_code = '';
        if ($evento_id) {
            $next_code = $this->participant->generateNextCode($evento_id);
        }

        $error = $_SESSION['error'] ?? '';
        unset($_SESSION['error']);

        include '../src/Views/participants/form.php';
    }

    public function create() {
        $evento_id = $_POST['evento_id'] ?? 0;
        
        // Validar evento
        if (!$this->event->findById($evento_id)) {
            $_SESSION['error'] = 'Evento não encontrado.';
            header('Location: /participantes/cadastrar');
            exit();
        }

        // Coletar dados do formulário
        $this->participant->evento_id = $evento_id;
        $this->participant->codigo_inscricao = trim($_POST['codigo_inscricao'] ?? '');
        $this->participant->nome_completo = trim($_POST['nome_completo'] ?? '');
        $this->participant->nome_credencial = trim($_POST['nome_credencial'] ?? $this->participant->nome_completo);
        $this->participant->rg = trim($_POST['rg'] ?? '');
        $this->participant->cpf = trim($_POST['cpf'] ?? '');
        $this->participant->genero = trim($_POST['genero'] ?? '');
        $this->participant->data_nascimento = !empty($_POST['data_nascimento']) ? $_POST['data_nascimento'] : null;
        $this->participant->estado = trim($_POST['estado'] ?? '');
        $this->participant->cidade = trim($_POST['cidade'] ?? '');
        $this->participant->cep = trim($_POST['cep'] ?? '');
        $this->participant->bairro = trim($_POST['bairro'] ?? '');
        $this->participant->endereco = trim($_POST['endereco'] ?? '');
        $this->participant->cargo = trim($_POST['cargo'] ?? '');
        $this->participant->telefone = trim($_POST['telefone'] ?? '');
        $this->participant->email = trim($_POST['email'] ?? '');

        // Validações
        $errors = [];

        if (empty($this->participant->codigo_inscricao)) {
            $this->participant->codigo_inscricao = $this->participant->generateNextCode($evento_id);
        }

        if ($this->participant->codeExists($this->participant->codigo_inscricao, $evento_id)) {
            $errors[] = 'Código de inscrição já existe neste evento.';
        }

        if (empty($this->participant->nome_completo)) {
            $errors[] = 'Nome completo é obrigatório.';
        }

        if (empty($this->participant->email)) {
            $errors[] = 'Email é obrigatório.';
        } elseif (!filter_var($this->participant->email, FILTER_VALIDATE_EMAIL)) {
            $errors[] = 'Email inválido.';
        } elseif ($this->participant->emailExists($this->participant->email, $evento_id)) {
            $errors[] = 'Email já cadastrado neste evento.';
        }

        if (!empty($errors)) {
            $_SESSION['error'] = implode('<br>', $errors);
            header('Location: /participantes/cadastrar?evento_id=' . $evento_id);
            exit();
        }

        if ($this->participant->create()) {
            $_SESSION['success'] = 'Participante cadastrado com sucesso!';
            header('Location: /participantes?evento_id=' . $evento_id);
        } else {
            $_SESSION['error'] = 'Erro ao cadastrar participante.';
            header('Location: /participantes/cadastrar?evento_id=' . $evento_id);
        }
        exit();
    }

    public function showImport() {
        $evento_id = $_GET['evento_id'] ?? null;
        
        // Buscar eventos
        $stmt = $this->event->read();
        $eventos = $stmt->fetchAll();

        $error = $_SESSION['error'] ?? '';
        $success = $_SESSION['success'] ?? '';
        unset($_SESSION['error'], $_SESSION['success']);

        include '../src/Views/participants/import.php';
    }

    public function import() {
        $evento_id = $_POST['evento_id'] ?? 0;
        
        // Validar evento
        if (!$this->event->findById($evento_id)) {
            $_SESSION['error'] = 'Evento não encontrado.';
            header('Location: /participantes/importar');
            exit();
        }

        // Verificar se arquivo foi enviado
        if (!isset($_FILES['excel_file']) || $_FILES['excel_file']['error'] !== UPLOAD_ERR_OK) {
            $_SESSION['error'] = 'Erro no upload do arquivo.';
            header('Location: /participantes/importar?evento_id=' . $evento_id);
            exit();
        }

        $file = $_FILES['excel_file'];
        $allowed_types = ['application/vnd.ms-excel', 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet'];
        
        if (!in_array($file['type'], $allowed_types)) {
            $_SESSION['error'] = 'Tipo de arquivo não permitido. Use apenas arquivos Excel (.xls ou .xlsx).';
            header('Location: /participantes/importar?evento_id=' . $evento_id);
            exit();
        }

        try {
            // Mover arquivo para diretório de uploads
            $upload_dir = '../uploads/excel/';
            if (!is_dir($upload_dir)) {
                mkdir($upload_dir, 0777, true);
            }
            
            $filename = uniqid() . '_' . $file['name'];
            $filepath = $upload_dir . $filename;
            
            if (!move_uploaded_file($file['tmp_name'], $filepath)) {
                throw new Exception('Erro ao salvar arquivo.');
            }

            // Ler arquivo Excel
            $spreadsheet = IOFactory::load($filepath);
            $worksheet = $spreadsheet->getActiveSheet();
            $data = $worksheet->toArray();

            // Remover cabeçalho
            $header = array_shift($data);
            
            // Mapear colunas (assumindo ordem padrão)
            $mapped_data = [];
            foreach ($data as $row) {
                if (empty(trim($row[0] ?? ''))) continue; // Pular linhas vazias
                
                $mapped_data[] = [
                    'nome_completo' => $row[0] ?? '',
                    'nome_credencial' => $row[1] ?? '',
                    'email' => $row[2] ?? '',
                    'telefone' => $row[3] ?? '',
                    'cargo' => $row[4] ?? '',
                    'rg' => $row[5] ?? '',
                    'cpf' => $row[6] ?? '',
                    'genero' => $row[7] ?? '',
                    'data_nascimento' => $row[8] ?? '',
                    'estado' => $row[9] ?? '',
                    'cidade' => $row[10] ?? '',
                    'cep' => $row[11] ?? '',
                    'bairro' => $row[12] ?? '',
                    'endereco' => $row[13] ?? ''
                ];
            }

            // Importar dados
            $result = $this->participant->importFromArray($mapped_data, $evento_id);

            // Limpar arquivo temporário
            unlink($filepath);

            if ($result['imported'] > 0) {
                $message = "Importação concluída! {$result['imported']} participantes importados.";
                if (!empty($result['errors'])) {
                    $message .= "<br><br>Erros encontrados:<br>" . implode('<br>', $result['errors']);
                }
                $_SESSION['success'] = $message;
            } else {
                $_SESSION['error'] = 'Nenhum participante foi importado.<br>' . implode('<br>', $result['errors']);
            }

        } catch (Exception $e) {
            $_SESSION['error'] = 'Erro ao processar arquivo: ' . $e->getMessage();
        }

        header('Location: /participantes?evento_id=' . $evento_id);
        exit();
    }

    public function edit() {
        $id = $_GET['id'] ?? 0;

        if (!$this->participant->findById($id)) {
            $_SESSION['error'] = 'Participante não encontrado.';
            header('Location: /participantes');
            exit();
        }

        // Buscar eventos
        $stmt = $this->event->read();
        $eventos = $stmt->fetchAll();

        $error = $_SESSION['error'] ?? '';
        unset($_SESSION['error']);

        include '../src/Views/participants/edit.php';
    }

    public function update() {
        $id = $_POST['id'] ?? 0;
        
        if (!$this->participant->findById($id)) {
            $_SESSION['error'] = 'Participante não encontrado.';
            header('Location: /participantes');
            exit();
        }

        $evento_id = $this->participant->evento_id;

        // Atualizar dados
        $this->participant->codigo_inscricao = trim($_POST['codigo_inscricao'] ?? '');
        $this->participant->nome_completo = trim($_POST['nome_completo'] ?? '');
        $this->participant->nome_credencial = trim($_POST['nome_credencial'] ?? $this->participant->nome_completo);
        $this->participant->rg = trim($_POST['rg'] ?? '');
        $this->participant->cpf = trim($_POST['cpf'] ?? '');
        $this->participant->genero = trim($_POST['genero'] ?? '');
        $this->participant->data_nascimento = !empty($_POST['data_nascimento']) ? $_POST['data_nascimento'] : null;
        $this->participant->estado = trim($_POST['estado'] ?? '');
        $this->participant->cidade = trim($_POST['cidade'] ?? '');
        $this->participant->cep = trim($_POST['cep'] ?? '');
        $this->participant->bairro = trim($_POST['bairro'] ?? '');
        $this->participant->endereco = trim($_POST['endereco'] ?? '');
        $this->participant->cargo = trim($_POST['cargo'] ?? '');
        $this->participant->telefone = trim($_POST['telefone'] ?? '');
        $this->participant->email = trim($_POST['email'] ?? '');

        // Validações
        $errors = [];

        if ($this->participant->codeExists($this->participant->codigo_inscricao, $evento_id, $id)) {
            $errors[] = 'Código de inscrição já existe neste evento.';
        }

        if (empty($this->participant->nome_completo)) {
            $errors[] = 'Nome completo é obrigatório.';
        }

        if (empty($this->participant->email)) {
            $errors[] = 'Email é obrigatório.';
        } elseif (!filter_var($this->participant->email, FILTER_VALIDATE_EMAIL)) {
            $errors[] = 'Email inválido.';
        } elseif ($this->participant->emailExists($this->participant->email, $evento_id, $id)) {
            $errors[] = 'Email já cadastrado neste evento.';
        }

        if (!empty($errors)) {
            $_SESSION['error'] = implode('<br>', $errors);
            header('Location: /participantes/editar?id=' . $id);
            exit();
        }

        if ($this->participant->update()) {
            $_SESSION['success'] = 'Participante atualizado com sucesso!';
        } else {
            $_SESSION['error'] = 'Erro ao atualizar participante.';
        }

        header('Location: /participantes?evento_id=' . $evento_id);
        exit();
    }

    public function delete() {
        $id = $_GET['id'] ?? 0;

        if (!$this->participant->findById($id)) {
            $_SESSION['error'] = 'Participante não encontrado.';
            header('Location: /participantes');
            exit();
        }

        $evento_id = $this->participant->evento_id;

        if ($this->participant->delete()) {
            $_SESSION['success'] = 'Participante excluído com sucesso!';
        } else {
            $_SESSION['error'] = 'Erro ao excluir participante.';
        }

        header('Location: /participantes?evento_id=' . $evento_id);
        exit();
    }
}

